
/*
 * \file
 * <!--
 * This file was part of BeRTOS and modified by Alejandro Santos.
 *
 * Bertos is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * As a special exception, you may use this file as part of a free software
 * library without restriction.  Specifically, if other files instantiate
 * templates or use macros or inline functions from this file, or you compile
 * this file and link it with other files to produce an executable, this
 * file does not by itself cause the resulting executable to be covered by
 * the GNU General Public License.  This exception does not however
 * invalidate any other reasons why the executable file might be covered by
 * the GNU General Public License.
 *
 * Copyright 2010 Develer S.r.l. (http://www.develer.com/)
 *
 * Copyright 2013 Alejandro Santos LU4EXT.
 *
 * -->
 *
 * \author Modified by Alejandro Santos, LU4EXT <alejolp@gmail.com>
 *
 * Based on the BeRTOS work by:
 *
 * \author Francesco Sacchi <batt@develer.com>
 * \author Luca Ottaviano <lottaviano@develer.com>
 * \author Daniele Basile <asterix@develer.com>
 *
 * \brief Arduino APRS Digipeater
 *
 * Breytingar gerðar af: Samúel Þór Hjaltalín Guðjónsson
 *
 */
// Köllum í eftirfarandi kóðasöfn
#include <cpu/irq.h>
#include <cfg/debug.h>

#include <net/afsk.h>
#include <net/ax25.h>

#include <drv/ser.h>
#include <drv/timer.h>

#include <stdio.h>
#include <string.h>

static Afsk afsk; // static breyta fyrir afsk
static AX25Ctx ax25; // static breyta fyrir ax25
static Serial ser; // static breyta fyrir serial samskipti

#define ADC_CH 0 // Við notum ADC inngang 0

/* Breytum þessu í rétt kallmerki.  */
static const char MYCALL[] = "TF3SUT"; // Kallmerki stafvarpa
static uint8_t MYCALL_SSID = 3; // SSID stafvarpa

/* Vitastrengur sem stafvarpinn sendir út með reglulegu millibili */
#define APRS_BEACON_MSG    "!6408.55N/02155.52W#>APRS Stafvarpi" // Vitastrengur
#define APRS_BEACON_TIME (1 * 60) // Bil milli vitasendinga (mín * sek)

/* Við breytum þessu ekki, apzbrt lætur aðra stafvarpa vita að við séum APRS BertOS. */
#define CALL_BERTOS_APRS "apzbrt" //

/*
 * Skrifum á seríal móttekin skilaboð
 */
static void message_callback(struct AX25Msg *msg)
{
    int i, k;

    static AX25Call tmp_path[AX25_MAX_RPT + 2];
    static uint8_t tmp_path_size;
    static const char* relay_calls[] = {"RELAY\x0", "WIDE\x0\x0", "TRACE\x0", 0};


#if 1
    kfile_printf(&ser.fd, "\n\nSRC[%.6s-%d], DST[%.6s-%d]\r\n", msg->src.call, msg->src.ssid, msg->dst.call, msg->dst.ssid);

    for (i = 0; i < msg->rpt_cnt; i++)
        kfile_printf(&ser.fd, "via: [%.6s-%d%s]\r\n", msg->rpt_lst[i].call, msg->rpt_lst[i].ssid, msg->rpt_lst[i].h_bit?"*":"");

    kfile_printf(&ser.fd, "DATA: %.*s\r\n", msg->len, msg->info);
#endif

    if (1) {
        uint8_t repeat = 0; // breyta fyrir endurvörpun
        uint8_t is_wide = 0, is_trace = 0; // breyta fyrir wide og trace

        msg->dst.ssid = 0; // færum skilaboð í dst.ssid
        msg->dst.h_bit = 0; // ef h bit er high, þá endurvörpum við
        memcpy(msg->dst.call, CALL_BERTOS_APRS, 6);

        tmp_path[0] = msg->dst;
        tmp_path[1] = msg->src;
// bætt við =< í stað <
        for (i = 0; i < msg->rpt_cnt; ++i) // Ef það eru stafir eftir í skilaboðum
            tmp_path[i + 2] = msg->rpt_lst[i];
        tmp_path_size = 2 + msg->rpt_cnt;

        /* Á að endurvarpa pakkanum?
         * http://wa8lmf.net/DigiPaths/ */

        /*
         * Ef path inniheldur WIDE,RELAY eða TRACE og er ekki búin með -n þá
         * endurvörpum við pakkanum
         */

        for (i = 2; i < tmp_path_size; ++i) {
            if (!tmp_path[i].h_bit) {
                AX25Call* c = &tmp_path[i];

                if ((memcmp(tmp_path[i].call, MYCALL, 6) == 0) && (tmp_path[i].ssid == MYCALL_SSID))
                {
                    repeat = 0;
                    break;
                }

                for (k=0; relay_calls[k]; ++k)
                {
                    if (memcmp(relay_calls[k], c->call, 6) == 0) {
                        repeat = 1;
                        tmp_path[i].h_bit = 1;
                        break;
                    }
                }

                if (repeat)
                    break;

                if (tmp_path[i].ssid > 0)
                {
                    is_wide = memcmp("WIDE", c->call, 4) == 0;
                    is_trace = memcmp("TRACE", c->call, 5) == 0;
                    // ef það er annaðhvort wide eða trace, þá endurvörpum við
                    if (is_wide || is_trace)
                    {
                        repeat = 1;
                        tmp_path[i].ssid--;
                        if (tmp_path[i].ssid == 0)
                            tmp_path[i].h_bit = 1;

                        /* Bætum við kallmerki stafvarpans */
//breytti is_trace í is_wide
                        if ((is_trace) || (is_wide) && (tmp_path_size < (AX25_MAX_RPT + 2)))
												{
                            for (k = tmp_path_size; k > i; --k) {
                                tmp_path[k] = tmp_path[k - 1];
                            }
                            memcpy(tmp_path[i].call, MYCALL, 6);
                            tmp_path[i].ssid = MYCALL_SSID;
                            tmp_path[i].h_bit = 1;
                            tmp_path_size++;
                            i++;
                        }

                        break;
                    }
                }
            } else
						{
                if ((memcmp(tmp_path[i].call, MYCALL, 6) == 0) && (tmp_path[i].ssid == MYCALL_SSID)) {
                    repeat = 1;
                    tmp_path[i].h_bit = 1;
                    break;
                }
            }
        }

        if ((memcmp(tmp_path[1].call, MYCALL, 6) == 0) && (tmp_path[i].ssid == MYCALL_SSID))
				{
            repeat = 0;
        }

        if (repeat)
				{
            ax25_sendVia(&ax25, tmp_path, tmp_path_size, msg->info, msg->len);
            kfile_print(&ser.fd, "REPEATED\n");
        } else {
            kfile_print(&ser.fd, "NOT REPEATED\n");
        }
    }
}

static void init(void)
{
    IRQ_ENABLE;
    kdbg_init();
    timer_init();

    /*
     * Ræsum AFSK afmótara. Skilgreiningar eru í hw_afsk.h
     */
    afsk_init(&afsk, ADC_CH, 0);
    /*
     * Ræsum AX.25
     */
    ax25_init(&ax25, &afsk.fd, message_callback);

#if 1
    /* Ræsum serial samskipti */
    ser_init(&ser, SER_UART0);
    ser_setbaudrate(&ser, 115200L); // Við notum 115200 baud rate
#endif
}
// Við viljum að vitamerki sendis sé sent út með path WIDE2-1, ekki WIDE1-1,WIDE2-2. AX25_CALL("", 0), AX25_CALL("", 0),
// Fjarlægjum því "AX25_CALL("wide1", 1), AX25_CALL("wide2", 2));" og breytum því í eftirfarandi
static AX25Call path[] = AX25_PATH(AX25_CALL(CALL_BERTOS_APRS, 0),  AX25_CALL("", 0),AX25_CALL("wide2", 1));

// Aðal fall byrjar
int main(void)
{
    init();
    ticks_t start = timer_clock(); // endursetjum teljara
    unsigned char x = 0;
    // setjum kallmerki og SSID í streng
    memcpy(path[1].call, MYCALL, 6);
    path[1].ssid = MYCALL_SSID;

    while (1)//while fall sem keyrir svo lengi sem Arduino er í gangi
    {
        /* Svo lengi sem CONFIG_AFSK_RXTIMEOUT er sett sem 0, þá pollum við ax25. */
        ax25_poll(&ax25);

        // Við köllum alltaf á þetta fall
#if 1
        /* Sendum skilaboð út á serial á 15s fresti */
        if (timer_clock() - start > ms_to_ticks(APRS_BEACON_TIME * 1000L))
        {
            kfile_printf(&ser.fd, "Beep %d\n", x++);
            start = timer_clock(); // endursetjum teljara
            // sendum út vitastreng
            ax25_sendVia(&ax25, path, countof(path), APRS_BEACON_MSG, sizeof(APRS_BEACON_MSG));
        }
#endif

    }
    return 0;
} // aðalfall endar
